<?php
// File: src/Controller/DonateController.php
namespace App\Controller;

use Cake\Controller\Controller;
use Stripe\Stripe;
use Stripe\Checkout\Session;
use Stripe\Exception\ApiErrorException;
use Cake\Core\Configure;
use Cake\Routing\Router;
use Cake\Mailer\Mailer;

class DonateController extends Controller
{
    public function initialize(): void
    {
        parent::initialize();
        Stripe::setApiKey(Configure::read('Stripe.secretKey'));
        $this->loadComponent('Flash');

    }

    public function beforeFilter(\Cake\Event\EventInterface $event)
    {
        parent::beforeFilter($event);

    }

    // Donation form page
    public function index()
    {
        // Render donation form view
        // Fetch the Donations table instance
        $donationsTable = $this->fetchTable('Donations');
 
        // Query to sum the 'amount' field in the Donations table
        $query = $donationsTable->find();
        $query->select([
            'total' => $query->func()->sum('amount')
        ]);
 
        // Get the sum from the first (and only) result
        $result = $query->first();
        $totalDonations = $result ? $result->total : 0;
 
        $campaignsTable = $this->fetchTable('Campaigns');
        $campaigns = $campaignsTable->find()->all();
 
        $totalsByCampaign = $donationsTable->find()
            ->select(['campaign_id', 'total' => $donationsTable->find()->func()->sum('amount')])
            ->where(['campaign_id IS NOT' => null])
            ->group('campaign_id')
            ->enableHydration(false)
            ->toArray();

        // Pass the total donations and campaigns to the view
        $this->set(compact('totalDonations', 'campaigns'));
        $this->set('totalsByCampaign', $totalsByCampaign);

    }
    public function createDonationSession()
    {
        $this->autoRender = false;        
        try {
            $selectedAmount = $this->request->getData('amount');
            if ($selectedAmount === 'custom') {
                $donationAmount = floatval($this->request->getData('custom_amount'));
            } else {
                $donationAmount = floatval($selectedAmount);
            }
            if ($donationAmount < 1) {
                return $this->response->withType('application/json')
                                      ->withStringBody(json_encode(['error' => 'Minimum donation is $1']));
            }
            $session = Session::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => 'aud',
                        'unit_amount' => $donationAmount * 100,
                        'product_data' => ['name' => 'Donation'],
                    ],
                    'quantity' => 1,
                ]],
                'metadata' => [
                    'campaign_id' => $this->request->getData('campaign_id') ?? 'unknown'
                ],
                'mode' => 'payment',
                'success_url' => Router::url(['controller' => 'Donate', 'action' => 'thankyou'], true),
                'cancel_url' => Router::url(['controller' => 'Donate', 'action' => 'cancel'], true),
            ]);
            return $this->response->withType('application/json')
                                  ->withStringBody(json_encode(['id' => $session->id]));
        } catch (\Throwable $e) {
            $this->log($e->getMessage(), 'error');
            return $this->response->withType('application/json')
                                  ->withStringBody(json_encode(['error' => $e->getMessage()]));
        }
    }

    // Successful donation handler
    public function success()
    {
        $this->Flash->success('Thank you for your donation!');
        return $this->redirect(['action' => 'index']);
    }

    // Cancelled donation handler
    public function cancel()
    {
        $this->Flash->warning('Donation was cancelled.');
        return $this->redirect(['action' => 'index']);
    }

// Internal method to process donation after payment
protected function _processSuccessfulDonation($paymentIntent)
{
    // Fetch the Donations table instance using fetchTable() (CakePHP 4.x approach)
    $donationsTable = $this->fetchTable('Donations');

    // Check if paymentIntent data is valid
    if (empty($paymentIntent) || empty($paymentIntent['amount'])) {
        $this->log('Invalid paymentIntent data: ' . json_encode($paymentIntent), 'error');
        return $this->response->withStatus(400); // Bad request
    }

    // Convert amount from cents to dollars
    $amount = $paymentIntent['amount'] / 100;
    // Use receipt_email as the default donor email
    $donorEmail = $paymentIntent['receipt_email'] ?? 'unknown@example.com';

    // Extract donor's name from billing_details
    $donorName = 'Anonymous';
    $chargesData = $paymentIntent['charges']['data'] ?? [];
    if (empty($chargesData)) {
        $latestChargeId = $paymentIntent['latest_charge'] ?? null;
        if ($latestChargeId) {
            try {
                // Retrieve charge details using latest_charge ID
                $charge = \Stripe\Charge::retrieve($latestChargeId);
                if ($charge && !empty($charge['billing_details']['name'])) {
                    $donorName = $charge['billing_details']['name'];
                }
            } catch (\Stripe\Exception\ApiErrorException $e) {
                // Log the error and use fallback name
                $this->log('Error retrieving charge details: ' . $e->getMessage(), 'error');
                $donorName = 'Anonymous';
            }
        }
    } else {
        if (!empty($chargesData[0]['billing_details']['name'])) {
            $donorName = $chargesData[0]['billing_details']['name'];
        }
    }

    // Handle missing charges data and fallback to latest_charge if necessary
    if (empty($chargesData)) {
        $latestChargeId = $paymentIntent['latest_charge'] ?? null;
        if ($latestChargeId) {
            try {
                // Retrieve charge details using latest_charge ID
                $charge = \Stripe\Charge::retrieve($latestChargeId);
                if ($charge && !empty($charge['billing_details']['email'])) {
                    $donorEmail = $charge['billing_details']['email'];
                }
            } catch (\Stripe\Exception\ApiErrorException $e) {
                // Log the error and use fallback email
                $this->log('Error retrieving charge details: ' . $e->getMessage(), 'error');
                $donorEmail = 'unknown@example.com';
            }
        }
    } else {
        if (!empty($chargesData[0]['billing_details']['email'])) {
            $donorEmail = $chargesData[0]['billing_details']['email'];
        }
    }

    // Extract campaign_id from paymentIntent metadata
    $campaignId = $paymentIntent['metadata']['campaign_id'] ?? null;
    $this->log('Received campaign_id: ' . $campaignId, 'debug');

    // Create a new donation entity with the fetched table
    $donationEntity = $donationsTable->newEntity([
        'amount' => $amount,
        'donor_email' => $donorEmail,
        'donor_name' => $donorName,
        'stripe_transaction_id' => $paymentIntent['id'] ?? 'unknown_txn_id',
        'status' => 'completed',
        'campaign_id' => $campaignId,
    ]);

    // Save the donation entity and log the result
    if (!$donationsTable->save($donationEntity)) {
        $this->log('Donation save failed: ' . json_encode($donationEntity->getErrors()), 'error');
    } else {
        $this->log('Donation saved successfully.', 'info');

        // Send thank you email to donor
        $mailer = new Mailer('default');
        $mailer->setTo($donorEmail)
            ->setSubject('Thank you for your donation!')
            ->setEmailFormat('html')
            ->deliver("
                <p>Dear {$donorName},</p>
                <p>Thank you for your generous donation of $" . number_format($amount, 2) . ".</p>
                <p>Your support means a lot to us!</p>
            ");
    }

    // Respond with 200 OK to signal successful processing
    return $this->response->withStatus(200);
}

// Stripe Webhook for donation confirmation
public function webhook()
{
    // Get the raw POST data
    $payload = file_get_contents('php://input');
    
    // Get the Stripe signature header from the request
    $sigHeader = $_SERVER['HTTP_STRIPE_SIGNATURE'];
    
    // Get the webhook secret from the configuration
    $endpointSecret = Configure::read('Stripe.webhookSecret');

    try {
        // Verify the webhook signature and construct the event
        $event = \Stripe\Webhook::constructEvent(
            $payload,
            $sigHeader,
            $endpointSecret
        );

        // Log the event type for debugging
        $this->log('Stripe Webhook received: ' . $event['type'], 'info');

        // Handle different event types
        switch ($event['type']) {
            case 'payment_intent.succeeded':
                // $paymentIntent = $event['data']['object'];
                // Process successful donation is handled in checkout.session.completed
                break;
            
            case 'checkout.session.completed':
                $session = $event['data']['object'];
                $this->log('Received campaign_id: ' . ($session['metadata']['campaign_id'] ?? 'N/A'), 'debug');
                
                try {
                    $campaignId = $session['metadata']['campaign_id'] ?? null;
                    if ($campaignId !== null) {
                        // Update metadata first
                        \Stripe\PaymentIntent::update($session['payment_intent'], [
                            'metadata' => ['campaign_id' => $campaignId],
                        ]);
                    }

                    // Re-fetch the updated PaymentIntent
                    $paymentIntent = \Stripe\PaymentIntent::retrieve($session['payment_intent']);

                    $this->_processSuccessfulDonation($paymentIntent);
                } catch (\Exception $e) {
                    $this->log('Failed to retrieve PaymentIntent for session completed: ' . $e->getMessage(), 'error');
                }
                break;

            case 'charge.updated':
                $charge = $event['data']['object'];
                // Log charge updates for tracking
                $this->log('Charge updated for: ' . $charge['id'], 'info');
                $stripeTxnId = $charge['payment_intent'] ?? null;
                $donorName = $charge['billing_details']['name'] ?? null;

                if ($stripeTxnId && $donorName) {
                    $donationsTable = $this->fetchTable('Donations');
                    $donation = $donationsTable->find()->where(['stripe_transaction_id' => $stripeTxnId])->first();
                    if ($donation) {
                        $donation->donor_name = $donorName;
                        if ($donationsTable->save($donation)) {
                            $this->log('Donor name updated for transaction ID: ' . $stripeTxnId, 'info');
                        } else {
                            $this->log('Failed to update donor name for transaction ID: ' . $stripeTxnId, 'error');
                        }
                    }
                }
                break;

            default:
                // Log unhandled events for future reference
                $this->log('Unhandled Stripe event type: ' . $event['type'], 'info');
                break;
        }

        // Respond with 200 OK to acknowledge the event was processed successfully
        return $this->response->withStatus(200);
    } catch (\UnexpectedValueException $e) {
        // Log error for invalid webhook payload
        $this->log('Invalid webhook payload: ' . $e->getMessage(), 'error');
        return $this->response->withStatus(400); // Bad request
    } catch (\Stripe\Exception\SignatureVerificationException $e) {
        // Log error for signature verification failure
        $this->log('Signature verification failed: ' . $e->getMessage(), 'error');
        return $this->response->withStatus(400); // Bad request
    } catch (\Stripe\Exception\ApiErrorException $e) {
        // Log Stripe API error
        $this->log('Stripe API Error: ' . $e->getMessage(), 'error');
        return $this->response->withStatus(500); // Internal server error
    } catch (\Exception $e) {
        // Log general error during webhook processing
        $this->log('Webhook processing failed: ' . $e->getMessage(), 'error');
        return $this->response->withStatus(500); // Internal server error
    }
}

// Thank you page shown after successful donation
public function thankyou()
{
    $this->viewBuilder()->setLayout('default');
}
}