<?php

declare(strict_types=1);

namespace App\Controller;

use App\Model\Table\ProgramsImagesTable;
use App\Model\Table\ProgramsVolunteersTable;
use Cake\Utility\Text;

/**
 * Programs Controller
 *
 * @property \App\Model\Table\ProgramsTable $Programs
 */
class ProgramsController extends AppController
{
    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    /**
     * @var \App\Model\Table\ProgramsImagesTable $ProgramsImagesTable
     * @property \App\Model\Table\ProgramsVolunteersTable $ProgramsVolunteers
     * /
     */

    private ProgramsImagesTable $ProgramsImagesTable;

    private ProgramsVolunteersTable $ProgramsVolunteers;

    public function initialize(): void
    {
        parent::initialize(); // TODO: Change the autogenerated stub
        $this->viewBuilder()->setLayout('adminlayout');
        $this->ProgramsImagesTable = $this->fetchTable('ProgramsImages');
        $this->ProgramsVolunteers = $this->fetchTable('ProgramsVolunteers');
    }

//    public function addvolunteer($id = null)
//    {
//        $program = $this->Programs->get($id, contain: ['Volunteers']);
//
//        if ($this->request->is(['patch', 'post', 'put'])) {
//            $program = $this->Programs->patchEntity($program, $this->request->getData());
//            if ($this->Programs->save($program)) {
//                $this->Flash->success(__('The program has been saved.'));
//
//                return $this->redirect(['action' => 'index']);
//            }
//            $this->Flash->error(__('The program could not be saved. Please, try again.'));
//
//        }
//
//        $volunteers = $this->Programs->Volunteers->find('list', limit: 200)->all();
//        $this->set(compact('program', 'volunteers'));
//
//
//    }

    public function remove($programId = null, $volunteerId = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $programVolunteer = $this->ProgramsVolunteers->find()
            ->where(['program_id' => $programId, 'volunteer_id' => $volunteerId])
            ->first();
        if ($this->ProgramsVolunteers->delete($programVolunteer)) {
            $this->Flash->success(__('The programs volunteer has been deleted.'));
        } else {
            $this->Flash->error(__('The programs volunteer could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    public function index()
    {
        // Fetch Programs with associated ProgramStatuses
        $programsTable = $this->fetchTable('Programs');

        // Handle search query
        $search = $this->request->getQuery('search');
        $query = $programsTable->find()
            ->contain(['ProgramStatuses']); // Fetch associated ProgramStatuses

        if (!empty($search)) {
            $query->where([
                'Programs.name LIKE' => '%' . $search . '%'
            ]);
        }

        // Paginate Programs
        $this->paginate = [
            'limit' => 1000,
            'order' => ['Programs.date' => 'desc'] // Order by date descending
        ];
        $programs = $this->paginate($query);

        // Set variables for the view
        $this->set(compact('programs'));
        $this->viewBuilder()->setLayout('adminlayout');
        $this->render('/Programs/index'); // Render the Programs index view
    }

    /**
     * View method
     *
     * @param string|null $id Program id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $program = $this->Programs->get($id, contain: ['ProgramStatuses', 'Students', 'Volunteers', 'ProgramsImages']);
        $this->set(compact('program'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $program = $this->Programs->newEmptyEntity();
        if ($this->request->is('post')) {
            $program = $this->Programs->patchEntity($program, $this->request->getData());
            if ($this->Programs->save($program)) {
                $imageFiles = $this->request->getData('image_files'); // This is now an array of images
                if (!empty($imageFiles)) {
                    foreach ($imageFiles as $imageFile) {
                        if ($imageFile->getClientFilename()) {
                            // Check the file type
                            $fileType = $imageFile->getClientMediaType();
                            $allowedTypes = ['image/jpeg', 'image/png'];

                            if (in_array($fileType, $allowedTypes)) {
                                // Define the image path
                                $imageName = time() . '-' . $imageFile->getClientFilename();
                                $imagePath = WWW_ROOT . 'uploads/img/' . $imageName;

                                // Move the uploaded file to the server
                                $imageFile->moveTo($imagePath);
                                $image_url = 'uploads/img/' . $imageName;
                                if ($image_url) {
                                    $programImage = $this->ProgramsImagesTable->newEmptyEntity();
                                    $programImage->image_name = $image_url;
                                    $programImage->program_id = $program->id;
                                    $this->ProgramsImagesTable->save($programImage);
                                }
                            }
                        }
                    }

                }

                // ✅ Add default ProgramsSection using the association
                $defaultSection = $this->Programs->ProgramsSections->newEntity([
                    'title' => 'Learn More',
                    'body' => '<h5> If you are interested, learn more by sending us an enquiry. </h5>',
                    'program_id' => $program->id
                ]);
                $this->Programs->ProgramsSections->save($defaultSection);

                $this->Flash->success(__('The program has been saved.'));
                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The program could not be saved. Please, try again.'));
        }
        $programStatuses = $this->Programs->ProgramStatuses->find('list', limit: 200)->all();
        $students = $this->Programs->Students->find('list', limit: 200)->all();
        $volunteers = $this->Programs->Volunteers->find('list', limit: 200)->all();
        $this->set(compact('program', 'programStatuses', 'students', 'volunteers'));
    }

    public function deleteimg($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $programsImage = $this->ProgramsImagesTable->get($id);
        if ($this->ProgramsImagesTable->delete($programsImage)) {
            $this->Flash->success(__('The programs image has been deleted.'));
        } else {
            $this->Flash->error(__('The programs image could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    private function uploadFile($imageFile): string
    {
        $imageFiles = $this->request->getData('image_files'); // This is now an array of images
        if (!empty($imageFiles)) {
            foreach ($imageFiles as $imageFile) {
                if ($imageFile->getClientFilename()) {
                    // Check the file type
                    $fileType = $imageFile->getClientMediaType();
                    $allowedTypes = ['image/jpeg', 'image/png'];

                    if (in_array($fileType, $allowedTypes)) {
                        // Define the image path
                        $imageName = time() . '-' . $imageFile->getClientFilename();
                        $imagePath = WWW_ROOT . 'img/uploads/' . $imageName;

                        // Move the uploaded file to the server
                        $imageFile->moveTo($imagePath);

                    }
                }
            }
        }
        return '';
    }


    /**
     * Edit method
     *
     * @param string|null $id Program id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
{
    $program = $this->Programs->get($id, contain: ['ProgramsImages', 'Students', 'Volunteers', 'ProgramStatuses']);

    if ($this->request->is(['patch', 'post', 'put'])) {
        // Prepare the data for Students and Volunteers
        $studentsData = $this->request->getData('students._ids');
        $volunteersData = $this->request->getData('volunteers._ids');

        // If no students are selected, set the Students association to an empty array
        if (empty($studentsData)) {
            $program->students = [];
        } else {
            // Otherwise, patch the Students association by passing only the IDs
            $program->students = $studentsData;
        }

        // If no volunteers are selected, set the Volunteers association to an empty array
        if (empty($volunteersData)) {
            $program->volunteers = [];
        } else {
            // Otherwise, patch the Volunteers association by passing only the IDs
            $program->volunteers = $volunteersData;
        }

        // Now patch the program entity (with the associated data for Students and Volunteers)
        $program = $this->Programs->patchEntity($program, $this->request->getData(), [
            'associated' => ['Students', 'Volunteers'] // Handle the associated Students and Volunteers
        ]);

        // Handle the removal of images
        if (!empty($this->request->getData('remove_images'))) {
            $removeImages = $this->request->getData('remove_images');
            foreach ($removeImages as $imageId) {
                $image = $this->ProgramsImagesTable->get($imageId);
                // Delete the image file from disk
                $imagePath = WWW_ROOT . $image->image_name;
                if (file_exists($imagePath)) {
                    unlink($imagePath); // Delete the image from the server
                }

                // Remove the image entity from the database
                $this->ProgramsImagesTable->delete($image);
            }
        }

        // Handle the new images uploaded via the file input
        $imageFiles = $this->request->getData('image_files');
        if (!empty($imageFiles)) {
            foreach ($imageFiles as $imageFile) {
                if ($imageFile->getClientFilename()) {
                    // Check the file type
                    $fileType = $imageFile->getClientMediaType();
                    $allowedTypes = ['image/jpeg', 'image/png'];

                    // Validate the file type
                    if (in_array($fileType, $allowedTypes)) {
                        // Define the image path and save the file
                        $imageName = time() . '-' . $imageFile->getClientFilename();
                        $imagePath = WWW_ROOT . 'uploads/img/' . $imageName;
                        $imageFile->moveTo($imagePath);

                        // Create a new ProgramsImages entity
                        $programImage = $this->ProgramsImagesTable->newEmptyEntity();
                        $programImage->image_name = 'uploads/img/' . $imageName;
                        $programImage->program_id = $program->id;

                        // Save the new image entity to the database
                        $this->ProgramsImagesTable->save($programImage);
                    }
                }
            }
        }

        // Save the updated program entity
        if ($this->Programs->save($program)) {
            $this->Flash->success(__('The program has been saved.'));
            return $this->redirect(['action' => 'index']);
        }

        $this->Flash->error(__('The program could not be saved. Please, try again.'));
    }

    // Load available students and volunteers
    $programStatuses = $this->Programs->ProgramStatuses->find('list', limit: 200)->all();
    $students = $this->Programs->Students->find('list', limit: 200)->all();
    $volunteers = $this->Programs->Volunteers->find('list', limit: 200)->all();

    $this->set(compact('program', 'programStatuses', 'students', 'volunteers'));
}







    /**
     * Delete method
     *
     * @param string|null $id Program id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $program = $this->Programs->get($id);
        if ($this->Programs->delete($program)) {
            $this->Flash->success(__('The program has been deleted.'));
        } else {
            $this->Flash->error(__('The program could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }


    public function downloadPdf($id = null)
    {
        // Start output buffering to prevent any output
        ob_start();

        try {
            // Fetch the program with all related data
            $program = $this->Programs->get($id, [
                'contain' => [
                    'Students',
                    'Volunteers.VolunteerStatuses',
                    'ProgramsImages'
                ]
            ]);

            // Initialize TCPDF
            $pdf = new \TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

            // Set document information
            $pdf->SetCreator(PDF_CREATOR);
            $pdf->SetAuthor('Your Application');
            $pdf->SetTitle('Program: ' . $program->name);
            $pdf->SetSubject('Program Details');

            // Remove default header/footer
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);

            // Set margins
            $pdf->SetMargins(15, 15, 15);
            $pdf->SetAutoPageBreak(true, 15);

            // Add a page
            $pdf->AddPage();

            // Generate HTML content for PDF
            $html = '<h1>Program Title: ' . h($program->name) . '</h1>';

            // Program Details
            $html .= '<h2>Program Details</h2>';
            $html .= '<table border="1" cellpadding="5">';
            $html .= '<tr><th>Name</th><td>' . h($program->name) . '</td></tr>';
            $html .= '<tr><th>Status</th><td>' . ($program->hasValue('program_status') ? h($program->program_status->name) : '') . '</td></tr>';
            $html .= '<tr><th>ID</th><td>' . $program->id . '</td></tr>';
            $html .= '<tr><th>Date</th><td>' . h($program->date) . '</td></tr>';
            $html .= '<tr><th>Time</th><td>' . h($program->time) . '</td></tr>';
            $html .= '</table>';

            // Students
            if (!empty($program->students)) {
                $html .= '<h2>Related Students</h2>';
                $html .= '<table border="1" cellpadding="5">';
                $html .= '<tr style="background-color: #e87532; color: white;">';
                $html .= '<th>Name</th><th>Year Level</th><th>School</th><th>Phone</th>';
                $html .= '</tr>';
                foreach ($program->students as $student) {
                    $html .= '<tr>';
                    $html .= '<td>' . h($student->first_name . ' ' . $student->last_name) . '</td>';
                    $html .= '<td>' . h($student->year_level) . '</td>';
                    $html .= '<td>' . h($student->school) . '</td>';
                    $html .= '<td><a href="tel:' . h($student->contact_phone) . '" style="color: #e87532; text-decoration: none;">' . h($student->contact_phone) . '</a></td>';
                    $html .= '</tr>';
                }
                $html .= '</table>';
            }

            // Volunteers
            if (!empty($program->volunteers)) {
                $html .= '<h2>Related Volunteers</h2>';
                $html .= '<table border="1" cellpadding="5" cellspacing="0" style="border-collapse: collapse; width: 100%;">';
                $html .= '<tr style="background-color: #e87532; color: white; text-align: left;">';
                $html .= '<th>First Name</th><th>Last Name</th><th>Email</th><th>Phone</th>';
                $html .= '<th>Postcode</th><th>Occupation</th><th>Description</th><th>Status</th>';
                $html .= '</tr>';

                foreach ($program->volunteers as $volunteer) {
                    $html .= '<tr>';
                    $html .= '<td>' . h($volunteer->first_name) . '</td>';
                    $html .= '<td>' . h($volunteer->last_name) . '</td>';
                    $html .= '<td><a href="mailto:' . h($volunteer->email) . '" style="color: #e87532; text-decoration: underline;">' . h($volunteer->email) . '</a></td>';
                    $html .= '<td><a href="tel:' . h($volunteer->phone) . '" style="color: #e87532; text-decoration: none;">' . h($volunteer->phone) . '</a></td>';
                    $html .= '<td>' . h($volunteer->postcode) . '</td>';
                    $html .= '<td>' . h($volunteer->occupation) . '</td>';
                    $html .= '<td>' . h($volunteer->description) . '</td>';
                    $html .= '<td>' . ($volunteer->hasValue('volunteer_status') ? h($volunteer->volunteer_status->name) : h($volunteer->volunteer_status_id)) . '</td>';
                    $html .= '</tr>';
                }
                $html .= '</table>';
            }

            // Images in table format
            $imagePaths = [];
            if (!empty($program->programs_images)) {

                $html .= '<h2>Related Images</h2>';
                foreach ($program->programs_images as $index => $image) {
                    $imagePath = WWW_ROOT . $image->image_name;
                    $jpgPath = '';

                    // Check if the image exists and is a PNG
                    if (file_exists($imagePath) && pathinfo($imagePath, PATHINFO_EXTENSION) == 'png') {
                        // Convert PNG to JPG to avoid transparency issues
                        $pngImage = imagecreatefrompng($imagePath);
                        $jpgPath = WWW_ROOT . 'temp_' . pathinfo($image->image_name, PATHINFO_FILENAME) . '.jpg';

                        // Create a white background image and copy the PNG onto it
                        $width = imagesx($pngImage);
                        $height = imagesy($pngImage);
                        $white = imagecreatetruecolor($width, $height);
                        imagefill($white, 0, 0, imagecolorallocate($white, 255, 255, 255));
                        imagecopy($white, $pngImage, 0, 0, 0, 0, $width, $height);

                        // Save the image as JPG
                        imagejpeg($white, $jpgPath, 90);

                        // Free up memory
                        imagedestroy($pngImage);
                        imagedestroy($white);



                        // Also store the path to add the actual image later
                        $imagePaths[] = ['path' => $jpgPath, 'is_temp' => true];

                    } else if (file_exists($imagePath)) {
                        // For non-PNG images (JPEG, GIF, etc.)
                        // Add more image details if available
                        if (property_exists($image, 'title') && !empty($image->title)) {
                            $html .= 'Title: ' . h($image->title) . '<br>';
                        }
                        if (property_exists($image, 'description') && !empty($image->description)) {
                            $html .= 'Description: ' . h($image->description) . '<br>';
                        }
                        $html .= '</td>';
                        $html .= '</tr>';

                        // Also store the path to add the actual image later
                        $imagePaths[] = ['path' => $imagePath, 'is_temp' => false];
                    }
                }

                $html .= '</table>';
            }

            // Write HTML to PDF
            $pdf->writeHTML($html, true, false, true, false, '');

            // Add images on a separate page if we have any
            if (!empty($imagePaths)) {
                $pdf->AddPage();
                $pdf->SetFont('helvetica', 'B', 16);
                $pdf->Cell(0, 10, 'Image Gallery', 0, 1, 'C');
                $pdf->SetFont('helvetica', '', 12);

                $x = 20;
                $y = 40;
                $maxHeight = 100;

                foreach ($imagePaths as $item) {
                    // Add each image to the PDF
                    $pdf->Image($item['path'], $x, $y, 60, 0, '', '', 'T', false, 300, '', false, false, 1);

                    // Move position for next image
                    $y += $maxHeight;

                    // If we're running out of page space, add a new page
                    if ($y > 240) {
                        $pdf->AddPage();
                        $y = 40;
                    }

                    // Clean up temp files if needed
                    if ($item['is_temp'] && file_exists($item['path'])) {
                        unlink($item['path']);
                    }
                }
            }

            // Clear any previous output
            ob_end_clean();

            // Output PDF
            $filename = 'program_' . Text::slug($program->name) . '.pdf';
            $pdf->Output($filename, 'D'); // 'D' forces download

            // Exit to prevent view rendering
            exit();

        } catch (\Exception $e) {
            // If an error occurs, clean buffer and show error
            ob_end_clean();
            $this->Flash->error('Error generating PDF: ' . $e->getMessage());
            return $this->redirect(['action' => 'view', $id]);
        }
    }


    public function editSection($id = null)
    {
        $program = $this->Programs->get($id, [
            'contain' => ['ProgramsSections']
        ]);

        if ($this->request->is(['post', 'put'])) {
            $sectionsData = $this->request->getData('sections');
            $newSections = [];

            foreach ($sectionsData as $section) {
                $cleanBody = $section['content'] ?? '';

                // ✅ Fix <a href="..."> links missing https://
                $cleanBody = preg_replace_callback(
                    '/<a\s+[^>]*href="(?!https?:\/\/)([^"]+)"[^>]*>/i',
                    function ($matches) {
                        $url = 'https://' . ltrim($matches[1], '/');
                        return str_replace($matches[1], $url, $matches[0]);
                    },
                    $cleanBody
                );

                $newSections[] = $this->Programs->ProgramsSections->newEntity([
                    'title' => $section['title'] ?? '',
                    'body' => $cleanBody,
                    'program_id' => $program->id
                ]);
            }

            $connection = $this->Programs->getConnection();
            $connection->begin();

            try {
                $this->Programs->ProgramsSections->deleteAll(['program_id' => $program->id]);

                if ($this->Programs->ProgramsSections->saveMany($newSections)) {
                    $connection->commit();
                    $this->Flash->success('Sections updated successfully.');
                    return $this->redirect(['action' => 'editSection', $program->id]);
                } else {
                    throw new \Exception('Save failed');
                }
            } catch (\Exception $e) {
                $connection->rollback();
                $this->Flash->error('Something went wrong while saving sections. Changes were not applied.');
            }
        }

        $this->set(compact('program'));
    }



}




