<?php
declare(strict_types=1);

namespace App\Controller;

use App\Model\Table\DocTable;
use App\Model\Table\ProgramsTable;
use App\Model\Table\VolunteerStatusesTable;
use Cake\I18n\FrozenTime;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpParser\Node\Scalar\String_;

/**
 * Volunteers Controller
 *
 * @property \App\Model\Table\VolunteersTable $Volunteers
 */
class VolunteersController extends AppController
{
    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */

    /**
     * @var \App\Model\Table\VolunteerStatusesTable $VolunteerStatusTables
     * @var \App\Model\Table\DocTable $DocTable
     * @var \App\Model\Table\ProgramsTable $ProgramTable
     */

    public function initialize(): void
    {
        parent::initialize();
        $this->loadComponent('Flash');

        $this->VolunteerStatusTables = $this->fetchTable('VolunteerStatuses');
        $this->DocTable = $this->fetchTable('Doc');
        $this->ProgramTable = $this->fetchTable('Programs');
    }

    private VolunteerStatusesTable $VolunteerStatusTables;
    private ProgramsTable $ProgramTable;
    private DocTable $DocTable;


    public function index()
    {
        $this->viewBuilder()->setLayout('adminlayout');
        $query = $this->Volunteers->find()
            ->contain(['VolunteerStatuses']);
        $volunteers = $this->paginate($query);
        $this->set(compact('volunteers'));
        // Fetch Volunteers with associated VolunteerStatuses
        $volunteersTable = $this->fetchTable('Volunteers');

        // Handle search query
        $search = $this->request->getQuery('search');
        $query = $volunteersTable->find()
            ->contain(['VolunteerStatuses']); // Fetch associated VolunteerStatuses

        if (!empty($search)) {
            $query->where([
                'OR' => [
                    'Volunteers.first_name LIKE' => '%' . $search . '%',
                    'Volunteers.last_name LIKE' => '%' . $search . '%'
                ]
            ]);
        }

        // Paginate Volunteers
        $this->paginate = [
            'limit' => 1000,
            'order' => ['Volunteers.registration_date' => 'desc'] // Order by registration_date descending
        ];
        $volunteers = $this->paginate($query);

        // Set variables for the view
        $this->set(compact('volunteers'));
        $this->viewBuilder()->setLayout('adminlayout');
        $this->render('/Volunteers/index'); // Render the Volunteers index view
    }


    public function generatecertificate()
    {
        $this->viewBuilder()->setLayout('adminlayout');
        $query = $this->Volunteers->find()
            ->contain(['VolunteerStatuses']);
        $volunteers = $this->paginate($query);
        $this->set(compact('volunteers'));
        // Fetch Volunteers with associated VolunteerStatuses
        $volunteersTable = $this->fetchTable('Volunteers');

        // Handle search query
        $search = $this->request->getQuery('search');
        $query = $volunteersTable->find()
            ->contain(['VolunteerStatuses']); // Fetch associated VolunteerStatuses

        if (!empty($search)) {
            $query->where([
                'OR' => [
                    'Volunteers.first_name LIKE' => '%' . $search . '%',
                    'Volunteers.last_name LIKE' => '%' . $search . '%'
                ]
            ]);
        }

        // Paginate Volunteers
        $this->paginate = [
            'limit' => 1000,
            'order' => ['Volunteers.registration_date' => 'desc'] // Order by registration_date descending
        ];
        $volunteers = $this->paginate($query);

        // Set variables for the view
        $this->set(compact('volunteers'));
        $this->viewBuilder()->setLayout('adminlayout');
        $this->render('/Volunteers/generatecertificate '); // Render the Volunteers index view
    }

    /**
     * View method
     *
     * @param string|null $id Volunteer id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)

    {
        $this->viewBuilder()->setLayout('adminlayout');
        $volunteer = $this->Volunteers->get($id, contain: ['VolunteerStatuses', 'Programs']);
        $this->set(compact('volunteer'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $this->viewBuilder()->setLayout('adminlayout');
        $volunteer = $this->Volunteers->newEmptyEntity();
        if ($this->request->is('post')) {
            $volunteer = $this->Volunteers->patchEntity($volunteer, $this->request->getData());
            if ($this->Volunteers->save($volunteer)) {
                $this->Flash->success(__('The volunteer has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The volunteer could not be saved. Please, try again.'));
        }
        $volunteerStatuses = $this->Volunteers->VolunteerStatuses->find('list', limit: 200)->all();
        $programs = $this->Volunteers->Programs->find('list', limit: 200)->all();
        $this->set(compact('volunteer', 'volunteerStatuses', 'programs'));
    }

    /**
     * Edit method
     *
     * @param string|null $id Volunteer id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        $volunteer = $this->Volunteers->get($id, contain: ['Programs']);
        $this->viewBuilder()->setLayout('adminlayout');
        if ($this->request->is(['patch', 'post', 'put'])) {
            $volunteer = $this->Volunteers->patchEntity($volunteer, $this->request->getData());
            if ($this->Volunteers->save($volunteer)) {
                $this->Flash->success(__('The volunteer has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The volunteer could not be saved. Please, try again.'));
        }
        $volunteerStatuses = $this->Volunteers->VolunteerStatuses->find('list', limit: 200)->all();
        $programs = $this->Volunteers->Programs->find('list', limit: 200)->all();
        $this->set(compact('volunteer', 'volunteerStatuses', 'programs'));
    }

    /**
     * Delete method
     * *
     * @param string|null $id Volunteer id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $volunteer = $this->Volunteers->get($id);
        if ($this->Volunteers->delete($volunteer)) {
            $this->Flash->success(__('The volunteer has been deleted.'));
        } else {
            $this->Flash->error(__('The volunteer could not be deleted. Please, try again.'));
        }
        return $this->redirect(['action' => 'index']);
    }

    public function generate(){

        if ($this->request->is('post')) {
            $insertedVolunteers = [];
            $failedVolunteers = [];
            $file = $this->request->getData('uploaded_file');

            if ($file && $file->getError() === UPLOAD_ERR_OK) {
                $filepath = $this->handleFileUpload($file);
                if (!$filepath) {
                    // Flash message is set in handleFileUpload
                    return $this->redirect(['action' => 'index']);
                }

                try {
                    // Check if it's a CSV to potentially adjust reader options
                    $fileExtension = strtolower(pathinfo($filepath, PATHINFO_EXTENSION));
                    if ($fileExtension === 'csv') {
                              $spreadsheet = IOFactory::load($filepath); // IOFactory::load usually detects fine
                    } else {
                        $spreadsheet = IOFactory::load($filepath);
                    }

                    $worksheet = $spreadsheet->getActiveSheet();

                    if ($this->checkCsvHeader1($worksheet) === false) { // Use new header check function
                        $this->Flash->error(__('Invalid or mismatched header row in the uploaded file. Please ensure the first row matches: volunteer_email, badges, certified, program_id1, program_id2, program_id3'));
                        // Clean up before redirecting
                        if ($filepath && file_exists($filepath)) {
                            unlink($filepath);
                        }
                        return $this->redirect(['action' => 'generatecertificate']);
                    }

                    $this->processSpreadsheetData1($worksheet, $insertedVolunteers, $failedVolunteers); // Use new processing function

                    // Store results in session for potential display (optional)
                     $this->request->getSession()->write('insertedVolunteers', $insertedVolunteers);
                     $this->request->getSession()->write('failedVolunteers', $failedVolunteers);
//
//                     \Cake\Error\debug($insertedVolunteers);
//                     \Cake\Error\debug($failedVolunteers);
                    if (!empty($insertedVolunteers) || !empty($failedVolunteers)) {

                        // Generate report only if there's something to report
//                        $this->generatePdfReport1($insertedVolunteers, $failedVolunteers);
//
//                        $this->Flash->success(__('File processed. Generating report...'));
//                        return $this->redirect(['controller' => 'Certificates', 'action' => 'index']);// Function handles exit/download

                        $this->Flash->success(__('Data Upload Successful.'));
                        $this->set(compact('insertedVolunteers', 'failedVolunteers'));
                        $this->viewBuilder()->setLayout('adminlayout');
                        return $this->render('review_volunteer_data'); // Points to `review_volunteer_data.ctp`

                        // Note: generatePdfReport calls exit(), so code below might not execute if PDF is generated
                    } else {
                        $this->Flash->warning(__('The uploaded file was empty or contained no processable data rows.'));
                    }
                    // Redirect occurs inside generatePdfReport or here if no report
                    $this->Flash->success(__('File processed. Generating report...'));
                    return $this->redirect(['controller' => 'Certificates', 'action' => 'index']);

                } catch (\PhpOffice\PhpSpreadsheet\Exception $e) { // Catch specific spreadsheet errors
                    $this->Flash->error(__('Error reading the spreadsheet file: ' . $e->getMessage()));
                } catch (\Exception $e) {
                    $this->Flash->error(__('An error occurred during processing: ' . $e->getMessage()));
                } finally {
                    // Clean up the uploaded file
                    if ($filepath && file_exists($filepath)) {
                        unlink($filepath);
                    }
                }
            } elseif ($file && $file->getError() !== UPLOAD_ERR_NO_FILE) {
                $this->Flash->error(__('Error during file upload: Code ' . $file->getError()));
            } else {
                $this->Flash->error(__('No file was uploaded. Please select a file.'));
            }
        }

        if (!$this->response->isDone()) { // Prevent headers already sent error if PDF was generated
            return $this->redirect(['action' => 'generatecertificate']);;
        }

    }

    /**
     * Upload method
     */
    public function upload()
    {
        $this->viewBuilder()->setLayout('adminlayout'); // Keep layout consistent
        $insertedVolunteers = [];
        $failedVolunteers = [];
        $filepath = null; // Initialize filepath

        if ($this->request->is('post')) {
            $file = $this->request->getData('uploaded_file');

            if ($file && $file->getError() === UPLOAD_ERR_OK) {
                $filepath = $this->handleFileUpload($file);
                if (!$filepath) {
                    // Flash message is set in handleFileUpload
                    return $this->redirect(['action' => 'index']);
                }

                try {
                    // Check if it's a CSV to potentially adjust reader options
                    $fileExtension = strtolower(pathinfo($filepath, PATHINFO_EXTENSION));
                    if ($fileExtension === 'csv') {
                        // Optional: Set CSV specific reader options if needed
                        // $reader = IOFactory::createReader('Csv');
                        // $reader->setInputEncoding('UTF-8'); // Example: Force UTF-8
                        // $spreadsheet = $reader->load($filepath);
                        $spreadsheet = IOFactory::load($filepath); // IOFactory::load usually detects fine
                    } else {
                        $spreadsheet = IOFactory::load($filepath);
                    }

                    $worksheet = $spreadsheet->getActiveSheet();

                    if ($this->checkCsvHeader($worksheet) === false) { // Use new header check function
                        $this->Flash->error(__('Invalid or mismatched header row in the uploaded file. Please ensure the first row matches: First Name, Last Name, Email, Phone, Postcode, Occupation, Status'));
                        // Clean up before redirecting
                        if ($filepath && file_exists($filepath)) {
                            unlink($filepath);
                        }
                        return $this->redirect(['action' => 'index']);
                    }

                    $this->processSpreadsheetData($worksheet, $insertedVolunteers, $failedVolunteers); // Use new processing function

                    // Store results in session for potential display (optional)
                    // $this->request->getSession()->write('insertedVolunteers', $insertedVolunteers);
                    // $this->request->getSession()->write('failedVolunteers', $failedVolunteers);

                    if (!empty($insertedVolunteers) || !empty($failedVolunteers)) {
                        $this->Flash->success(__('File processed. Generating report...'));
                        // Generate report only if there's something to report
                        $this->generatePdfReport($insertedVolunteers, $failedVolunteers); // Function handles exit/download
                        // Note: generatePdfReport calls exit(), so code below might not execute if PDF is generated
                    } else {
                        $this->Flash->warning(__('The uploaded file was empty or contained no processable data rows.'));
                    }
                    // Redirect occurs inside generatePdfReport or here if no report
                    return $this->redirect(['action' => 'index']);

                } catch (\PhpOffice\PhpSpreadsheet\Exception $e) { // Catch specific spreadsheet errors
                    $this->Flash->error(__('Error reading the spreadsheet file: ' . $e->getMessage()));
                } catch (\Exception $e) {
                    $this->Flash->error(__('An error occurred during processing: ' . $e->getMessage()));
                } finally {
                    // Clean up the uploaded file
                    if ($filepath && file_exists($filepath)) {
                        unlink($filepath);
                    }
                }
            } elseif ($file && $file->getError() !== UPLOAD_ERR_NO_FILE) {
                $this->Flash->error(__('Error during file upload: Code ' . $file->getError()));
            } else {
                $this->Flash->error(__('No file was uploaded. Please select a file.'));
            }
        } else {
            // If not POST, just render the index view which should contain the upload form
            // No need to redirect here, let the index action handle rendering
            // return $this->redirect(['action' => 'index']); // Avoid redirect loop
        }

        // Set variables for the view if needed (e.g., if rendering upload form directly)
        // $this->set(compact('insertedVolunteers', 'failedVolunteers'));
        // If the upload logic is on the index page, redirecting back to index is correct.
        // If 'upload' has its own view, remove the final redirect.
        // Assuming the form is on the index page:
        if (!$this->response->isDone()) { // Prevent headers already sent error if PDF was generated
            return $this->redirect(['action' => 'index']);
        }
    }

    private function processSpreadsheetData($worksheet, &$insertedVolunteers, &$failedVolunteers)
    {
        // Use toArray() for potentially easier row/column access, especially for CSV
        // null, true, true, true => read null values, calculate formulas, format values, read headings
        $dataArray = $worksheet->toArray('', true, true, false); // Get raw cell values

        // Remove header row
        array_shift($dataArray);

        foreach ($dataArray as $rowIndex => $row) {
            // $rowIndex is 0-based here after array_shift, representing actual row number - 2
            $actualRowNum = $rowIndex + 2; // For user feedback if needed
            $this->processVolunteerCsvRow($row, $actualRowNum, $insertedVolunteers, $failedVolunteers);
        }
    }
    private function processSpreadsheetData1($worksheet, &$insertedVolunteers, &$failedVolunteers)
    {
        // Use toArray() for potentially easier row/column access, especially for CSV
        // null, true, true, true => read null values, calculate formulas, format values, read headings
        $dataArray = $worksheet->toArray('', true, true, false); // Get raw cell values

        // Remove header row
        array_shift($dataArray);

        foreach ($dataArray as $rowIndex => $row) {
            // $rowIndex is 0-based here after array_shift, representing actual row number - 2
            $actualRowNum = $rowIndex + 2; // For user feedback if needed
            $this->processVolunteerCsvRow1($row, $actualRowNum, $insertedVolunteers, $failedVolunteers);
        }
    }

    private function processVolunteerCsvRow1(array $row, int $rowNum, &$insertedVolunteers, &$failedVolunteers)
    {
        // Column indices based on $expectedHeader = ['First Name', 'Last Name', 'Email', 'Phone', 'Postcode', 'Occupation', 'Status'];
        // Indices:                                  0             1            2        3          4           5           6
        $volunteeremail = $this->handleNullValues($row[0] ?? null);
        $badges = $this->handleNullValues($row[1] ?? null);
        $certified = $this->handleNullValues($row[2] ?? null);
        $program_id1 = $this->handleNullValues($row[3] ?? null);
        $program_id2 = $this->handleNullValues($row[4] ?? null);
        $program_id3 = $this->handleNullValues($row[5] ?? null);

//        \Cake\Error\debug($volunteer);

        // Basic validation: Check if essential fields like email are present
//        \Cake\Error\debug($program_id1);
        if (empty($volunteeremail) || empty($badges) || empty($certified) || empty($program_id1)){
            $failedVolunteers[] = [
                'data' => [
                    'email' => $volunteeremail,
                    'badges' => $badges,
                    'certified' => $certified,
                    'program_id1' => $program_id1,
                    'program_id2' => $program_id2,
                    '$program_id3'=>$program_id3
                ],
                'reason' => "Skipped row {$rowNum}: Missing required fields (email, badges, certified status, or program)."
            ];
            return;
        }

        // Check for existing volunteer by email
        $existingVolunteer = $this->Volunteers->find('all')->where(['email' => $volunteeremail])->first();
        if (!$existingVolunteer) {
            $failedVolunteers[] = [
                'data' => [
                    'email' => $volunteeremail,
                    'badges' => $badges,
                    'certified' => $certified,
                    'program_id1' => $program_id1,
                    'program_id2' => $program_id2,
                    '$program_id3'=>$program_id3
                ],
                'reason' => "Skipped row {$rowNum}: Volunteer does not exist"
            ];
            return;
        }
//        \Cake\Error\debug($existingVolunteer);
        // Find volunteer status ID by name
        $program_id1 = (int)$program_id1;
       if(!empty($program_id2)) $program_id2 = (int)$program_id2;
        if(!empty($program_id3)) $program_id3 = (int)$program_id3;
//        \Cake\Error\debug($program_id2);
        if($program_id1==0 || $program_id2==0 || $program_id3==0){
            $failedVolunteers[] = [
                'data' => [
                    'email' => $volunteeremail,
                    'badges' => $badges,
                    'certified' => $certified,
                    'program_id1' => $program_id1,
                    'program_id2' => $program_id2,
                    '$program_id3'=>$program_id3
                ],
                'reason' => "Skipped row {$rowNum}: Program does not exist"
            ];
            return;
        }
        $program1=$this->ProgramTable->find()->where(['id' => $program_id1])->first();
        $program2=$this->ProgramTable->find()->where(['id' => $program_id2])->first();
        $program3=$this->ProgramTable->find()->where(['id' => $program_id3])->first();

        if(!$program1 ){
            $failedVolunteers[] = [
                'data' => [
                    'email' => $volunteeremail,
                    'badges' => $badges,
                    'certified' => $certified,
                    'program_id1' => $program_id1,
                    'program_id2' => $program_id2,
                    '$program_id3'=>$program_id3
                ],
                'reason' => "Skipped row {$rowNum}: Program does not exist"
            ];
            return;
        }





        $docTable= $this->DocTable->newEmptyEntity();
        $docTable->docuniqueid = $this->generateRandomString(20); // Example unique ID
        $docTable->volunteer = $existingVolunteer->first_name . ' ' . $existingVolunteer->last_name;
        $docTable->name = 'd'; // Example document name
        $docTable->badges = $badges; // Example badges
        $docTable->certified = $certified; // Example certified status (boolean)
        $docTable->issue_date = FrozenTime::now();
        $docTable->program_name1 = $program1->name; // Example program name
        $docTable->program_name2 = $program2?->name ?? 'N/A'; // Example program name
        $docTable->program_name3 = $program3->name ?? 'N/A'; // Example program name
        if($this->DocTable->save($docTable)){
//            \Cake\Error\debug($docTable);
            $insertedVolunteers[] = [
                'docuniqueid' => $docTable->docuniqueid, // Unique ID for the document
                'volunteer' => $docTable->volunteer, // Volunteer full name
                'name' => $docTable->name, // Document name
                'badges' => $docTable->badges, // Badges
                'certified' => $docTable->certified, // Certified status
                'issue_date' => $docTable->issue_date->format('Y-m-d H:i:s'), // Issue date formatted
                'program_name1' => $docTable->program_name1, // Program name 1
                'program_name2' => $docTable->program_name2, // Program name 2
                'program_name3' => $docTable->program_name3, // Program name 3
            ];
        }else{
            $errors = $docTable->getErrors();// Get validation errors
//            \Cake\Error\debug($errors);
            $failedVolunteers[] = [
                'data' => [
                    'email' => $volunteeremail,
                    'badges' => $badges,
                    'certified' => $certified,
                    'program_id1' => $program_id1,
                    'program_id2' => $program_id2,
                    'program_id3' => $program_id3,
                ],
                'reason' =>  "Skipped row {$rowNum}:  {$errors}"
                ];

        }


    }

    private function generateRandomString($length = 20)
    {
        $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, strlen($characters) - 1)];
        }
        return $randomString;
    }



    private function processVolunteerCsvRow(array $row, int $rowNum, &$insertedVolunteers, &$failedVolunteers)
    {
        // Column indices based on $expectedHeader = ['First Name', 'Last Name', 'Email', 'Phone', 'Postcode', 'Occupation', 'Status'];
        // Indices:                                  0             1            2        3          4           5           6
        $firstName = $this->handleNullValues($row[0] ?? null);
        $lastName = $this->handleNullValues($row[1] ?? null);
        $email = $this->handleNullValues($row[2] ?? null);
        $phone = $this->handleNullValues($row[3] ?? null);
        $postcode = $this->handleNullValues($row[4] ?? null);
        $occupation = $this->handleNullValues($row[5] ?? null);
        $volunteerStatusName = $this->handleNullValues($row[6] ?? null);

        // Basic validation: Check if essential fields like email are present
        if (empty($email)) {
            $failedVolunteers[] = [
                'data' => [
                    'first_name' => $firstName, 'last_name' => $lastName, 'email' => $email,
                    'phone' => $phone, 'postcode' => $postcode, 'occupation' => $occupation,
                    'volunteer_status_name' => $volunteerStatusName, // Store name provided
                ],
                'reason' => "Skipped row {$rowNum}: Email is missing.",
            ];
            return; // Skip this row
        }

        // Check for existing volunteer by email
        $existingVolunteer = $this->Volunteers->find('all')->where(['email' => $email])->first();
        if ($existingVolunteer) {
            $failedVolunteers[] = [
                'data' => [
                    'first_name' => $firstName, 'last_name' => $lastName, 'email' => $email,
                    'phone' => $phone, 'postcode' => $postcode, 'occupation' => $occupation,
                    'volunteer_status_name' => $volunteerStatusName, // Store name provided
                ],
                'reason' => "Skipped row {$rowNum}: Email already exists.",
            ];
            return; // Skip this row
        }
//        \Cake\Error\debug($existingVolunteer);
        // Find volunteer status ID by name
        $statusId = null;
        if (!empty($volunteerStatusName)) {
            $statusEntity = $this->VolunteerStatusTables->find()
                ->where(['name' => $volunteerStatusName])
                ->first();
            if ($statusEntity) {
                $statusId = $statusEntity->id;
            } else {
                // Status name provided but not found in DB
                $failedVolunteers[] = [
                    'data' => [
                        'first_name' => $firstName, 'last_name' => $lastName, 'email' => $email,
                        'phone' => $phone, 'postcode' => $postcode, 'occupation' => $occupation,
                        'volunteer_status_name' => $volunteerStatusName, // Store name provided
                    ],
                    'reason' => "Skipped row {$rowNum}: Volunteer status '{$volunteerStatusName}' not found.",
                ];
                return; // Skip this row
            }
        } else {
            // Status name is empty in the CSV - decide if this is allowed (e.g., defaults to null or a specific status)
            // Option 1: Fail if status is required
            $failedVolunteers[] = [
                'data' => [
                    'first_name' => $firstName, 'last_name' => $lastName, 'email' => $email,
                    'phone' => $phone, 'postcode' => $postcode, 'occupation' => $occupation,
                    'volunteer_status_name' => '',
                ],
                'reason' => "Skipped row {$rowNum}: Volunteer status is missing.",
            ];
            return; // Skip this row
            // Option 2: Allow null status_id if the database column permits it
            // $statusId = null; // Already initialized to null, so just proceed
        }


        // If we reached here, email is unique and status ID was found (or handled)
        $result = $this->saveVolunteerData($firstName, $lastName, $email, $phone, $postcode, $occupation, $statusId);

        if ($result['status'] === 'success') {
            $insertedVolunteers[] = $result['volunteer']; // Store the saved entity
        } else {
            $failedVolunteers[] = [
                'data' => $result['data'], // Use data prepared for saving
                'reason' => "Failed to save row {$rowNum}: " . ($result['errors'] ? json_encode($result['errors']) : 'Unknown save error'),
            ];
        }
    }

    private function saveVolunteerData($first_name, $last_name, $email, $phone, $postcode, $occupation, $volunteer_status_id)
    {
        $dataToSave = [
            'first_name' => $first_name,
            'last_name' => $last_name,
            'email' => $email,
            // Handle phone: Ensure it's integer or null. Remove non-digits if necessary.
            'phone' => $phone === '' ? null : (int)preg_replace('/\D/', '', $phone), // Example: Strip non-digits
            'postcode' => $postcode,
            'occupation' => $occupation,
            'description'=>"",
            // Handle status ID: Ensure it's integer or null
            'volunteer_status_id' => $volunteer_status_id === null ? null : (int)$volunteer_status_id,
            // 'description' field removed as it's not in the new CSV header
        ];

        $volunteer = $this->Volunteers->newEntity($dataToSave);

        if ($this->Volunteers->save($volunteer)) {
            return ['status' => 'success', 'volunteer' => $volunteer, 'data' => $dataToSave, 'errors' => null];
        } else {
            // Log errors for debugging
            \Cake\Log\Log::error('Volunteer save failed: ' . print_r($volunteer->getErrors(), true) . ' Data: ' . print_r($dataToSave, true));
            return ['status' => 'error', 'volunteer' => null, 'data' => $dataToSave, 'errors' => $volunteer->getErrors()];
        }
    }

    private function checkCsvHeader1($worksheet): bool
    {
        // Expected headers for CSV import
        $expectedHeader = ["volunteer_email","badges","certified","program_id1","program_id2","program_id3"];

        $headerRow = 1;
        $actualHeader = [];
        $highestColumn = $worksheet->getHighestColumn(); // e.g., 'G'
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);

        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $cellValue = $worksheet->getCell(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col) . $headerRow)->getValue();
            $actualHeader[] = trim((string)$cellValue);
        }


        // Compare only up to the number of expected headers
        $actualHeaderSlice = array_slice($actualHeader, 0, count($expectedHeader));
//        \Cake\Error\debug($actualHeader);
//        \Cake\Error\debug($expectedHeader);
//        \Cake\Error\debug($actualHeaderSlice === $expectedHeader);
        return $actualHeaderSlice === $expectedHeader;
    }

    private function checkCsvHeader($worksheet): bool
    {
        // Expected headers for CSV import
        $expectedHeader = ['First Name', 'Last Name', 'Email', 'Phone', 'Postcode', 'Occupation', 'Status'];

        $headerRow = 1;
        $actualHeader = [];
        $highestColumn = $worksheet->getHighestColumn(); // e.g., 'G'
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);

        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $cellValue = $worksheet->getCell(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col) . $headerRow)->getValue();
            $actualHeader[] = trim((string)$cellValue);
        }


        // Compare only up to the number of expected headers
        $actualHeaderSlice = array_slice($actualHeader, 0, count($expectedHeader));
        return $actualHeaderSlice === $expectedHeader;
    }
    public function export()
    {
        $start = $this->request->getQuery('start_date');
        $end = $this->request->getQuery('end_date');

        $query = $this->Volunteers->find()
            ->contain(['VolunteerStatuses'])
            ->select([
                'first_name',
                'last_name',
                'email',
                'phone',
                'postcode',
                'occupation',
                'status' => 'VolunteerStatuses.name',
                'registration_date'
            ]);

        if ($start && $end) {
            $query->where([
                'Volunteers.registration_date >=' => $start . ' 00:00:00',
                'Volunteers.registration_date <=' => $end . ' 23:59:59',
            ]);
        }

        $volunteers = $query->toArray();

        $filename = sprintf(
            'volunteers_%s_to_%s.csv',
            date('d-M-y', strtotime($start ?: 'now')),
            date('d-M-y', strtotime($end ?: 'now'))
        );

        $this->response = $this->response->withDownload($filename);

        $headers = ['First Name', 'Last Name', 'Email', 'Phone', 'Postcode', 'Occupation', 'Status'];

        $this->set(compact('volunteers', 'headers'));
        $this->viewBuilder()
            ->setClassName('CsvView.Csv')
            ->setOption('serialize', 'volunteers')
            ->setOption('header', $headers);
    }

    public function preview()
    {
        $this->viewBuilder()->setLayout('adminlayout');
        $start = $this->request->getQuery('start_date');
        $end = $this->request->getQuery('end_date');

        $query = $this->Volunteers->find()
            ->contain(['VolunteerStatuses'])
            ->select([
                'first_name', 'last_name', 'email', 'phone', 'postcode', 'occupation',
                'status' => 'VolunteerStatuses.name',
                'registration_date'
            ])
            ->order(['Volunteers.registration_date' => 'DESC']);

        if ($start && $end) {
            $query->where([
                'Volunteers.registration_date >=' => "$start 00:00:00",
                'Volunteers.registration_date <=' => "$end 23:59:59",
            ]);
        }

        // Paginate so you don’t overload the page
        $volunteers = $this->paginate($query);

        $this->set(compact('volunteers', 'start', 'end'));
    }

    public function overview()
    {
        $this->viewBuilder()->setLayout('adminlayout');
        $start = $this->request->getQuery('start_date');
        $end = $this->request->getQuery('end_date');

        $query = $this->Volunteers->find()
            ->contain(['VolunteerStatuses'])
            ->order(['Volunteers.registration_date' => 'DESC']);

        if ($start && $end) {
            $query->where([
                'Volunteers.registration_date >=' => $start . ' 00:00:00',
                'Volunteers.registration_date <=' => $end . ' 23:59:59',
            ]);
        }

        $volunteers = $this->paginate($query);

        $this->set(compact('volunteers', 'start', 'end'));
    }

    private function checkHeader($worksheet): bool
    {
//        $expectedHeader = "first_name\tlast_name\temail\tphone\tpostcode\toccupation\tvolunteer_status_id";

        $expectedHeader = "first_name\tlast_name\temail\tphone\tpostcode\toccupation\tvolunteer_status\tdescription";
        // Get the header row as a string
        $headerRow = 1;
        $actualHeader = '';
        $columns = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H'];

        foreach ($columns as $index => $column) {
            $cellValue = $worksheet->getCell($column . $headerRow)->getValue() ?? '';
            $actualHeader .= $cellValue;
            if ($index < count($columns) - 1) {
                $actualHeader .= "\t";
            }
        }
        $expectedHeader = preg_replace('/\s+/', '', $expectedHeader);  // Remove all whitespace (spaces, tabs, newlines) from expected header
        $actualHeader = preg_replace('/\s+/', '', $actualHeader);  // Remove all whitespace from actual header

//        \Cake\Error\debug($actualHeader);
//        \Cake\Error\debug($expectedHeader);

        // Validate headers
        if (trim($actualHeader) !== $expectedHeader) {
            return false;
        }

//        $this->Flash->error($actualHeader);
        return true;
    }

    private function handleFileUpload($file)
    {
        $allowedTypes = [
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'text/csv'
        ];

        if (in_array($file->getClientMediaType(), $allowedTypes)) {
            $filename = $file->getClientFilename();
            $filepath = WWW_ROOT . 'uploads' . DS . time() . '-' . $filename;
            $file->moveTo($filepath);
            return $filepath;
        } else {
            $this->Flash->error(__('Invalid file type.'));
            return false;
        }
    }

    private function handleNullValues($value)
    {
        return $value ?? '';
    }


    private function generatePdfReport($insertedVolunteers, $failedVolunteers)
    {
        ob_start();

        try {
            $pdf = new \TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

            $pdf->SetCreator(PDF_CREATOR);
            $pdf->SetAuthor('Your Application');
            $pdf->SetTitle('Volunteer Upload Report');
            $pdf->SetSubject('Upload Results');

            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);

            $pdf->SetMargins(15, 15, 15);
            $pdf->SetAutoPageBreak(true, 15);

            $pdf->AddPage();

            $html = '<h1>Volunteer Upload Report</h1>';
            $html .= '<p>Date: ' . date('Y-m-d H:i:s') . '</p>';

            // Successfully Inserted Volunteers
            $html .= '<h2>Successfully Uploaded Volunteers (' . count($insertedVolunteers) . ')</h2>';
            if (!empty($insertedVolunteers)) {
                $html .= '<table border="1" cellpadding="5" cellspacing="0" style="border-collapse: collapse; width: 100%;">';
                $html .= '<tr style="background-color: #4CAF50; color: white;">';
                $html .= '<th>First Name</th><th>Last Name</th><th>Email</th><th>Phone</th>';
                $html .= '<th>Postcode</th><th>Occupation</th><th>Status ID</th>';
                $html .= '</tr>';

                foreach ($insertedVolunteers as $volunteer) {
                    $html .= '<tr>';
                    $html .= '<td>' . h($volunteer->first_name) . '</td>';
                    $html .= '<td>' . h($volunteer->last_name) . '</td>';
                    $html .= '<td>' . h($volunteer->email) . '</td>';
                    $html .= '<td>' . h($volunteer->phone) . '</td>';
                    $html .= '<td>' . h($volunteer->postcode) . '</td>';
                    $html .= '<td>' . h($volunteer->occupation) . '</td>';
                    $html .= '<td>' . h($volunteer->volunteer_status_id) . '</td>';
                    $html .= '</tr>';
                }
                $html .= '</table>';
            } else {
                $html .= '<p>No volunteers were successfully uploaded.</p>';
            }

            // Failed Volunteers
            $html .= '<h2>Failed Uploads (' . count($failedVolunteers) . ')</h2>';
            if (!empty($failedVolunteers)) {
                $html .= '<table border="1" cellpadding="5" cellspacing="0" style="border-collapse: collapse; width: 100%;">';
                $html .= '<tr style="background-color: #f44336; color: white;">';
                $html .= '<th>First Name</th><th>Last Name</th><th>Email</th><th>Phone</th>';
                $html .= '<th>Postcode</th><th>Occupation</th><th>Status ID</th><th>Reason</th>';
                $html .= '</tr>';

                foreach ($failedVolunteers as $failed) {
                    // Ensure $data is an object with all fields
                    $data = is_array($failed['data']) ? (object)$failed['data'] : $failed['data'];
                    $html .= '<tr>';
                    $html .= '<td>' . h($data->first_name ?? '') . '</td>';
                    $html .= '<td>' . h($data->last_name ?? '') . '</td>';
                    $html .= '<td>' . h($data->email ?? '') . '</td>';
                    $html .= '<td>' . h($data->phone ?? '') . '</td>';
                    $html .= '<td>' . h($data->postcode ?? '') . '</td>';
                    $html .= '<td>' . h($data->occupation ?? '') . '</td>';
                    $html .= '<td>' . h($data->volunteer_status_id ?? '') . '</td>';
                    $html .= '<td>' . h($failed['reason']) . '</td>';
                    $html .= '</tr>';
                }
                $html .= '</table>';
            } else {
                $html .= '<p>No upload failures recorded.</p>';
            }

            $pdf->writeHTML($html, true, false, true, false, '');

            ob_end_clean();

            $filename = 'volunteer_upload_report_' . date('Ymd_His') . '.pdf';
            $pdf->Output($filename, 'D');
            exit();
        } catch (\Exception $e) {
            ob_end_clean();
            $this->Flash->error('Error generating PDF: ' . $e->getMessage());
            return $this->redirect(['action' => 'index']);
        }
    }

    public function downloadPdf()
    {
        $insertedVolunteers = unserialize(base64_decode($this->request->getData('insertedVolunteers')));
        $failedVolunteers = unserialize(base64_decode($this->request->getData('failedVolunteers')));

        $this->generatePdfReport1($insertedVolunteers, $failedVolunteers); // Will output and exit
    }


    private function generatePdfReport1($insertedVolunteers, $failedVolunteers)
    {
        ob_start();


        try {
            $pdf = new \TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

            $pdf->SetCreator(PDF_CREATOR);
            $pdf->SetAuthor('Your Application');
            $pdf->SetTitle('Certificate Generation Report');
            $pdf->SetSubject('Certificate Results');

            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);

            $pdf->SetMargins(15, 15, 15);
            $pdf->SetAutoPageBreak(true, 15);

            $pdf->AddPage();

            $html = '<h1>Certificate Generation Report</h1>';
            $html .= '<p>Date: ' . date('Y-m-d H:i:s') . '</p>';

            // Successfully Inserted Volunteers
            $html .= '<h2>Successfully Uploaded Volunteers (' . count($insertedVolunteers) . ')</h2>';
            if (!empty($insertedVolunteers)) {
                $html .= '<table border="1" cellpadding="5" cellspacing="0" style="border-collapse: collapse; width: 100%;">';
                $html .= '<tr style="background-color: #4CAF50; color: white;">';
                $html .= '<th>Certificate ID</th><th>VOlunteer Name </th><th>Badges</th><th>Certified</th>';
                $html .= '<th>Program Name1</th><th>Program Name2</th><th>Program Name3</th>';
                $html .= '</tr>';

                foreach ($insertedVolunteers as $certificate) {
                    $html .= '<tr>';
                    $html .= '<td>' . h($certificate['docuniqueid']) . '</td>';
                    $html .= '<td>' . h($certificate['volunteer']) . '</td>';
                    $html .= '<td>' . h($certificate['badges']) . '</td>';
                    $html .= '<td>' . h($certificate['certified']) . '</td>';
                    $html .= '<td>' . h($certificate['program_name1']) . '</td>';
                    $html .= '<td>' . h($certificate['program_name2']) . '</td>';
                    $html .= '<td>' . h($certificate['program_name3']) . '</td>';
                    $html .= '</tr>';
                }
                $html .= '</table>';
            } else {
                $html .= '<p>No volunteers were successfully uploaded.</p>';
            }

            // Failed Volunteers
            $html .= '<h2>Failed Uploads (' . count($failedVolunteers) . ')</h2>';
            if (!empty($failedVolunteers)) {
                $html .= '<table border="1" cellpadding="5" cellspacing="0" style="border-collapse: collapse; width: 100%;">';
                $html .= '<tr style="background-color: #f44336; color: white;">';
                $html .= '<th>Volunteer Email</th><th>Badges</th><th>Certified</th><th>Program ID1</th>';
                $html .= '<th>Program ID2</th><th>Program ID3</th><th>Reason</th>';
                $html .= '</tr>';

                foreach ($failedVolunteers as $failed) {
                    // Ensure $data is an object with all fields
                    $data = is_array($failed['data']) ? (object)$failed['data'] : $failed['data'];
                    $html .= '<tr>';
                    $html .= '<td>' . h($data->email ?? '') . '</td>';
                    $html .= '<td>' . h($data->badges ?? '') . '</td>';
                    $html .= '<td>' . h($data->certified ?? '') . '</td>';
                    $html .= '<td>' . h($data->program_id1 ?? '') . '</td>';
                    $html .= '<td>' . h($data->program_id2 ?? '') . '</td>';
                    $html .= '<td>' . h($data->program_id3 ?? '') . '</td>';
                    $html .= '<td>' . h($data->volunteer_status_id ?? '') . '</td>';
                    $html .= '<td>' . h($failed['reason']) . '</td>';
                    $html .= '</tr>';
                }
                $html .= '</table>';
            } else {
                $html .= '<p>No upload failures recorded.</p>';
            }

            $pdf->writeHTML($html, true, false, true, false, '');

            ob_end_clean();

            $filename = 'volunteer_upload_report_' . date('Ymd_His') . '.pdf';
            $pdf->Output($filename, 'D');
            exit();
        } catch (\Exception $e) {
            ob_end_clean();
            $this->Flash->error('Error generating PDF: ' . $e->getMessage());
            return $this->redirect(['action' => 'generate']);
        }
    }



}
