import React from 'react';
import { z } from 'zod';

type BaseZodDictionary = {
    [name: string]: z.AnyZodObject;
};
type DocumentBlocksDictionary<T extends BaseZodDictionary> = {
    [K in keyof T]: {
        schema: T[K];
        Component: (props: z.infer<T[K]>) => JSX.Element;
    };
};
type BlockConfiguration<T extends BaseZodDictionary> = {
    [TType in keyof T]: {
        type: TType;
        data: z.infer<T[TType]>;
    };
}[keyof T];

/**
 * @param blocks Main DocumentBlocksDictionary
 * @returns React component that can render a BlockConfiguration that is compatible with blocks
 */
declare function buildBlockComponent<T extends BaseZodDictionary>(blocks: DocumentBlocksDictionary<T>): ({ type, data }: BlockConfiguration<T>) => React.JSX.Element;

/**
 *
 * @param blocks Main DocumentBlocksDictionary
 * @returns zod schema that can parse arbitary objects into a single BlockConfiguration
 */
declare function buildBlockConfigurationSchema<T extends BaseZodDictionary>(blocks: DocumentBlocksDictionary<T>): z.ZodEffects<z.ZodDiscriminatedUnion<"type", any>, BlockConfiguration<T>, any>;

/**
 * Identity function to type a DocumentBlocksDictionary
 * @param blocks Main DocumentBlocksDictionary
 * @returns typed DocumentBlocksDictionary
 */
declare function buildBlockConfigurationDictionary<T extends BaseZodDictionary>(blocks: DocumentBlocksDictionary<T>): DocumentBlocksDictionary<T>;

export { type BlockConfiguration, type DocumentBlocksDictionary, buildBlockComponent, buildBlockConfigurationDictionary, buildBlockConfigurationSchema };
